#!/usr/bin/env python3
"""
Aplicación Flask para detección de chalecos - VERSIÓN HOSTING
Optimizada para hosting con Python (cPanel, etc.)
"""

import os
import cv2
import base64
import numpy as np
from flask import Flask, render_template, request, jsonify
import io
from PIL import Image
import logging

# Configurar logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

app = Flask(__name__)

def detect_chaleco_colors(image_data):
    """
    Detección basada en colores para chalecos
    Optimizada para hosting compartido
    """
    try:
        # Decodificar imagen base64
        image_bytes = base64.b64decode(image_data.split(',')[1])
        image = Image.open(io.BytesIO(image_bytes))
        
        # Convertir a numpy array
        frame = cv2.cvtColor(np.array(image), cv2.COLOR_RGB2BGR)
        
        # Redimensionar para procesamiento más rápido
        height, width = frame.shape[:2]
        if width > 600:  # Más pequeño para hosting
            scale = 600 / width
            new_width = 600
            new_height = int(height * scale)
            frame = cv2.resize(frame, (new_width, new_height))
        
        # Convertir a HSV para mejor detección de colores
        hsv = cv2.cvtColor(frame, cv2.COLOR_BGR2HSV)
        
        # Rangos de color para chalecos
        # Amarillo
        lower_yellow = np.array([20, 100, 100])
        upper_yellow = np.array([30, 255, 255])
        
        # Naranja
        lower_orange = np.array([10, 100, 100])
        upper_orange = np.array([20, 255, 255])
        
        # Verde fluorescente
        lower_green = np.array([40, 100, 100])
        upper_green = np.array([80, 255, 255])
        
        # Crear máscaras
        mask_yellow = cv2.inRange(hsv, lower_yellow, upper_yellow)
        mask_orange = cv2.inRange(hsv, lower_orange, upper_orange)
        mask_green = cv2.inRange(hsv, lower_green, upper_green)
        
        # Combinar todas las máscaras
        mask = cv2.bitwise_or(mask_yellow, mask_orange)
        mask = cv2.bitwise_or(mask, mask_green)
        
        # Operaciones morfológicas para limpiar la máscara
        kernel = np.ones((3,3), np.uint8)  # Kernel más pequeño para hosting
        mask = cv2.morphologyEx(mask, cv2.MORPH_OPEN, kernel)
        mask = cv2.morphologyEx(mask, cv2.MORPH_CLOSE, kernel)
        
        # Encontrar contornos
        contours, _ = cv2.findContours(mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)
        
        detections = []
        result_frame = frame.copy()
        
        for contour in contours:
            area = cv2.contourArea(contour)
            if area > 300:  # Umbral más bajo para hosting
                x, y, w, h = cv2.boundingRect(contour)
                
                # Calcular confianza
                aspect_ratio = w / h
                confidence = min(area / 3000, 0.95)
                
                # Ajustar confianza según la forma
                if 0.3 < aspect_ratio < 3.0:
                    confidence *= 1.1
                
                # Determinar si es chaleco
                class_name = "con_chaleco" if confidence > 0.3 else "sin_chaleco"
                
                detections.append({
                    "class": class_name,
                    "confidence": min(confidence, 0.95),
                    "bbox": [x, y, x+w, y+h]
                })
                
                # Dibujar resultado
                color = (0, 255, 0) if class_name == "con_chaleco" else (0, 0, 255)
                cv2.rectangle(result_frame, (x, y), (x+w, y+h), color, 2)
                cv2.putText(result_frame, f"{class_name}: {confidence:.2f}", 
                           (x, y-10), cv2.FONT_HERSHEY_SIMPLEX, 0.4, color, 1)
        
        # Si no hay detecciones
        if not detections:
            detections.append({
                "class": "sin_chaleco",
                "confidence": 0.6,
                "bbox": [0, 0, frame.shape[1], frame.shape[0]]
            })
            cv2.putText(result_frame, "Sin chaleco detectado", 
                       (30, 30), cv2.FONT_HERSHEY_SIMPLEX, 0.7, (0, 0, 255), 1)
        
        # Convertir imagen resultado a base64
        _, buffer = cv2.imencode('.jpg', result_frame, [cv2.IMWRITE_JPEG_QUALITY, 80])
        annotated_image = base64.b64encode(buffer).decode('utf-8')
        
        return {
            "detections": detections,
            "annotated_image": f"data:image/jpeg;base64,{annotated_image}",
            "success": True,
            "method": "color_detection_hosting"
        }
        
    except Exception as e:
        logger.error(f"Error al procesar imagen: {e}")
        return {"error": str(e)}

@app.route('/')
def index():
    """Página principal"""
    return render_template('index.html')

@app.route('/detect', methods=['POST'])
def detect():
    """Endpoint para detección de chalecos"""
    try:
        data = request.get_json()
        if 'image' not in data:
            return jsonify({"error": "No se proporcionó imagen"}), 400
        
        result = detect_chaleco_colors(data['image'])
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Error en endpoint de detección: {e}")
        return jsonify({"error": str(e)}), 500

@app.route('/health')
def health():
    """Endpoint de salud"""
    return jsonify({
        "status": "healthy",
        "model_loaded": True,
        "message": "Aplicación funcionando en hosting",
        "hosting": True
    })

@app.route('/info')
def info():
    """Información sobre la aplicación"""
    return jsonify({
        "name": "Detección de Chalecos - Hosting",
        "version": "1.0.0-hosting",
        "model_loaded": True,
        "description": "Aplicación web para detectar chalecos en hosting compartido",
        "hosting": True
    })

if __name__ == '__main__':
    logger.info("Aplicación iniciada en hosting")
    port = int(os.environ.get('PORT', 5000))
    app.run(host='0.0.0.0', port=port, debug=False)
